/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: RS232ConfigDlg.cpp 13422 2019-12-09 09:50:47Z Robert $:

  Description:
    Default Dialog

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2019-12-09  Updated file header to actual layout
    2011-08-08  Updated dialog item enable/disable handling
    2009-11-27  Review
    2009-07-28  initial version

**************************************************************************************/

/*****************************************************************************/
/*! \file RS232ConfigDlg.cpp                                                     
*   Configuration dialog for the RS232 connector                             */
/*****************************************************************************/

#include "stdafx.h"
#include "Windowsx.h"
#include "commctrl.h"

#include "RS232ConfigDlg.h"
#include "RS232Config.h"

#ifdef WIN64
  #include "resource_x64.h"
#else
  #include "resource_x32.h"
#endif

#include "netXConnectorErrors.h"

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_RS232 netX RS232 Connector                    */
/*! \{                                                                       */
/*****************************************************************************/

extern HANDLE      g_hModule;     /*!< Global DLL module handle */

CRS232ConfigDlg* CRS232ConfigDlg::s_cInstance = 0;

/////////////////////////////////////////////////////////////////////////////
/// Constructor
/////////////////////////////////////////////////////////////////////////////
CRS232ConfigDlg::CRS232ConfigDlg()
:m_hDialogWnd(NULL)
,m_cRS232Config()
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CRS232ConfigDlg::~CRS232ConfigDlg() 
{
}

/////////////////////////////////////////////////////////////////////////////
/// Get object instance (singleton)
/////////////////////////////////////////////////////////////////////////////
CRS232ConfigDlg* CRS232ConfigDlg::GetInstance()
{
  if( !s_cInstance )
    s_cInstance = new CRS232ConfigDlg();
  return s_cInstance;
}

/////////////////////////////////////////////////////////////////////////////
/// Create dialog window
///   \param hWndParent        Parent window
///   \param szConfig          Configuration file
///   \return Handle to created default dialog
/////////////////////////////////////////////////////////////////////////////
HWND CRS232ConfigDlg::Create ( HWND hWndParent, const char* szConfig)
{
  m_cRS232Config.Lock();

  /* Configuration string supplied ? */
  if (szConfig)
  {
    m_cRS232Config.ParseString( szConfig);
  }
  
  m_hDialogWnd = CreateDialog((HINSTANCE)g_hModule, MAKEINTRESOURCE(IDD_DIALOG) , hWndParent, (DLGPROC)DialogProcStatic);

  /* Assign options to dialog items */
  PopulateComboBoxes();
  
  /* Initialize Up-Down Controls */
  InitUpDownControl();
 
  /* Display interface configuaration in dialog */
  ShowSelInterfaceConfig();

  /* Display layer configuaration in dialog */
  ShowLayerConfig();
  
  m_cRS232Config.Unlock();

  return m_hDialogWnd;
}

/////////////////////////////////////////////////////////////////////////////
/// Destroy dialog window
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::Destroy( void)
{
  if (s_cInstance)
  {
    INT_PTR iRet = 0;
    EndDialog(m_hDialogWnd, iRet );
    delete s_cInstance;
    s_cInstance = NULL;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Get current dialog configuration as string
///   \param csRet  String buffer
///   \return NXCON_NO_ERROR if succeeded
/////////////////////////////////////////////////////////////////////////////
long CRS232ConfigDlg::GetConfiguration( CString& csRet)
{
  long lRet = NXCON_NO_ERROR;

  m_cRS232Config.Lock();
  lRet = m_cRS232Config.CreateString( csRet);
  m_cRS232Config.Unlock();
  
  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Set current dialog configuration as string
/// \param  szConfig Configuration string
/// \return NXCON_NO_ERROR if successful
/////////////////////////////////////////////////////////////////////////////
long CRS232ConfigDlg::SetConfiguration( const char* szConfig)
{
  long lRet      = NXCON_NO_ERROR;

  m_cRS232Config.Lock();
  if (NXCON_NO_ERROR == (lRet = m_cRS232Config.ParseString( szConfig)))
  {
    HWND hIntfList = GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF);
    
    /* Remove all interfaces from combobox */
    ComboBox_ResetContent(hIntfList);

    /* Load interfaces to combobox */
    for (CRS232Config::iterator iterIntf = m_cRS232Config.begin(); iterIntf != m_cRS232Config.end(); ++iterIntf) 
      ComboBox_AddString(hIntfList, *iterIntf);  
    ComboBox_SetCurSel(hIntfList, 0);

    ShowSelInterfaceConfig();
    ShowLayerConfig();
  }
  m_cRS232Config.Unlock();
  
  return lRet;
}


/////////////////////////////////////////////////////////////////////////////
/// Initiate Up/Down controls for edit boxes (Send/Reset/KeepAlive Timeout )
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::InitUpDownControl( void)
{
  HWND hItemWnd = NULL;
  long lMin     = 0;
  long lMax     = 0;

  /* Initialize up-down control for send timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_SEND);
  lMin = m_cRS232Config.GetMinVal(eINTERFACE, NXCON_COMMON_SENDTIMEOUT);
  lMax = m_cRS232Config.GetMaxVal(eINTERFACE, NXCON_COMMON_SENDTIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_SEND), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);

  /* Initialize up-down control for reset timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_RESET);
  lMin = m_cRS232Config.GetMinVal(eINTERFACE, NXCON_COMMON_RESETTIMEOUT);
  lMax = m_cRS232Config.GetMaxVal(eINTERFACE, NXCON_COMMON_RESETTIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_RESET), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);

  /* Initialize up-down control for keep alive timeout */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_SPIN_KEEPALIVE);
  lMin = m_cRS232Config.GetMinVal(eINTERFACE, NXCON_COMMON_KEEPALIVETIMEOUT);
  lMax = m_cRS232Config.GetMaxVal(eINTERFACE, NXCON_COMMON_KEEPALIVETIMEOUT);
  SendMessage(hItemWnd, UDM_SETBUDDY, (WPARAM)GetDlgItem(m_hDialogWnd,IDC_EDIT_KEEPALIVE), 0);
  SendMessage(hItemWnd, UDM_SETRANGE32, lMin, lMax);
}

/////////////////////////////////////////////////////////////////////////////
/// Populate the Comboboxes used by the dialog (Interface, Baudrate ...)
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::PopulateComboBoxes( void)
{
  int  iItem    = 0;
  HWND hItemWnd = NULL;

  /* Load baudrate options to combobox */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_COMBO_BAUDRATE);
  iItem = ComboBox_AddString(hItemWnd, "9.6 kBit/s");  
  ComboBox_SetItemData(hItemWnd, iItem, CBR_9600);
  iItem = ComboBox_AddString(hItemWnd, "19.2 kBit/s");  
  ComboBox_SetItemData(hItemWnd, iItem, CBR_19200);
  iItem = ComboBox_AddString(hItemWnd, "38.4 kBit/s");  
  ComboBox_SetItemData(hItemWnd, iItem, CBR_38400);
  iItem = ComboBox_AddString(hItemWnd, "57.6 kBit/s");  
  ComboBox_SetItemData(hItemWnd, iItem, CBR_57600);
  iItem = ComboBox_AddString(hItemWnd, "115.2  kBit/s");  
  ComboBox_SetItemData(hItemWnd, iItem, CBR_115200);

  /* Load parity options to combobox */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_COMBO_PARITY);
  iItem = ComboBox_AddString(hItemWnd, "No Parity");  
  ComboBox_SetItemData(hItemWnd, iItem, NOPARITY);
  iItem = ComboBox_AddString(hItemWnd, "Odd Parity");  
  ComboBox_SetItemData(hItemWnd, iItem, ODDPARITY);
  iItem = ComboBox_AddString(hItemWnd, "Even Parity");  
  ComboBox_SetItemData(hItemWnd, iItem, EVENPARITY);
  iItem = ComboBox_AddString(hItemWnd, "Mark Parity");  
  ComboBox_SetItemData(hItemWnd, iItem, MARKPARITY);
  iItem = ComboBox_AddString(hItemWnd, "Space Parity");  
  ComboBox_SetItemData(hItemWnd, iItem, SPACEPARITY);

  /* Load stopbits options to combobox */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_COMBO_STOPBITS);
  iItem = ComboBox_AddString(hItemWnd, "1 Stopbit");  
  ComboBox_SetItemData(hItemWnd, iItem, ONESTOPBIT);
  iItem = ComboBox_AddString(hItemWnd, "1.5 Stopbits");  
  ComboBox_SetItemData(hItemWnd, iItem, ONE5STOPBITS);
  iItem = ComboBox_AddString(hItemWnd, "2 Stopbits");  
  ComboBox_SetItemData(hItemWnd, iItem, TWOSTOPBITS);

  /* Load stopbits options to combobox */
  hItemWnd = GetDlgItem(m_hDialogWnd, IDC_COMBO_BYTESIZE);
  iItem = ComboBox_AddString(hItemWnd, "7 Byte");  
  ComboBox_SetItemData(hItemWnd, iItem, 7);
  iItem = ComboBox_AddString(hItemWnd, "8 Byte");  
  ComboBox_SetItemData(hItemWnd, iItem, 8);

  /* Load interfaces to combobox */
  for (CRS232Config::iterator iterIntf = m_cRS232Config.begin(); iterIntf != m_cRS232Config.end(); ++iterIntf) 
    ComboBox_AddString(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), *iterIntf);  
  ComboBox_SetCurSel(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), 0);  
}

/////////////////////////////////////////////////////////////////////////////
/// Set the layer dialog item sate (enable/disable)
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::SetLayerDlgItemState( void)
{
  if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE)))
  {
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), TRUE);

    /* Set the item states of the interface dialog */
    SetInterfaceDlgItemState();

  }else
  {
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), FALSE);

    /* Disable all interface elements from the sub window */
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_BAUDRATE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_BYTESIZE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_PARITY), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_STOPBITS), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), FALSE);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Set interface dialog item sate (enable/disable)
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::SetInterfaceDlgItemState( void)
{
  /* Read interface excluded from checkbox and store to config object */
  if (BST_CHECKED != Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE)))
  {
    /* Interface configuration available, so enable controls for interface configuration */
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_BAUDRATE), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_BYTESIZE), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_PARITY), TRUE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_STOPBITS), TRUE);

  }else
  {
    /* Disable all elements from the sub window */
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_BAUDRATE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_BYTESIZE), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_PARITY), FALSE);
    EnableWindow(GetDlgItem(m_hDialogWnd, IDC_COMBO_STOPBITS), FALSE);
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Read interface configuration from dialog controls and store values 
/// in the layer and interface configuration Map
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::ReadFromDialogItems( NX_CONF_KEY_SCOPE_E eKeyScope, CONFIGKEY_MAP& cmKeyMap)
{
  char szValue[MAX_PATH] = "";  
  int  iItem             = 0;
  HWND hItemWnd          = NULL;

  switch (eKeyScope)
  {
    case eINTERFACE:
       /* Read keep alive timeout value from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_COMMON_KEEPALIVETIMEOUT, szValue));
      
      /* Read send timeout value from editbox and store to config object */      
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_COMMON_SENDTIMEOUT, szValue));
      
      /* Read reset timeout value from editbox and store to config object */
      Edit_GetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), szValue, sizeof(szValue)/sizeof(*szValue));
      cmKeyMap.insert(make_pair(NXCON_COMMON_RESETTIMEOUT, szValue));
      
      /* Read baudrate value from combobox and store to config object */
      hItemWnd = GetDlgItem(m_hDialogWnd,IDC_COMBO_BAUDRATE);
      iItem    = ComboBox_GetCurSel(hItemWnd);
      sprintf(szValue,"%d", ComboBox_GetItemData(hItemWnd, iItem));
      cmKeyMap.insert(make_pair(NXCON_RS232KEY_BAUDRATE, szValue));

      /* Read bytesize value from combobox and store to config object */
      hItemWnd = GetDlgItem(m_hDialogWnd,IDC_COMBO_BYTESIZE);
      iItem    = ComboBox_GetCurSel(hItemWnd);
      sprintf(szValue,"%d", ComboBox_GetItemData(hItemWnd, iItem));
      cmKeyMap.insert(make_pair(NXCON_RS232KEY_BYTESIZE, szValue));

      /* Read parity value from combobox and store to config object */
      hItemWnd = GetDlgItem(m_hDialogWnd,IDC_COMBO_PARITY);
      iItem    = ComboBox_GetCurSel(hItemWnd);
      sprintf(szValue,"%d", ComboBox_GetItemData(hItemWnd, iItem));
      cmKeyMap.insert(make_pair(NXCON_RS232KEY_PARITY, szValue));

      /* Read stopbits value from combobox and store to config object */
      hItemWnd = GetDlgItem(m_hDialogWnd,IDC_COMBO_STOPBITS);
      iItem    = ComboBox_GetCurSel(hItemWnd);
      sprintf(szValue,"%d", ComboBox_GetItemData(hItemWnd, iItem));
      cmKeyMap.insert(make_pair(NXCON_RS232KEY_STOPBITS, szValue));

      /* Read interface excluded from checkbox and store to config object */
      if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE)))
        cmKeyMap.insert(make_pair(NXCON_COMMON_EXCLUDE, "1"));
      else
        cmKeyMap.insert(make_pair(NXCON_COMMON_EXCLUDE, "0"));

      break;
      
    case eLAYER:
      /* Read connector enabled from checkbox and store to config object */
      if (BST_CHECKED == Button_GetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE)))
        cmKeyMap.insert(make_pair(NXCON_COMMON_ENABLED, "1"));
      else
        cmKeyMap.insert(make_pair(NXCON_COMMON_ENABLED, "0"));
      break;
    default:
      return;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Write interface configuration from the interface map to the dialog controls
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::WriteToDialogItems( NX_CONF_KEY_SCOPE_E eKeyScope, CONFIGKEY_MAP& cmKeyMap)
{
  CONFIGKEY_MAP::iterator iterKeys = cmKeyMap.end();
  
  switch (eKeyScope)
  {
    case eINTERFACE:
      /* Read keep alive timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_KEEPALIVETIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_KEEPALIVE), iterKeys->second);

      /* Read send timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_SENDTIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_SEND), iterKeys->second);

      /* Read reset timeout value from config object and write to editbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_RESETTIMEOUT)))
        Edit_SetText(GetDlgItem(m_hDialogWnd, IDC_EDIT_RESET), iterKeys->second);

      /* Read baudrate value from config object and write to combobox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_RS232KEY_BAUDRATE)))
        SetComboBox ( IDC_COMBO_BAUDRATE, iterKeys->second);

      /* Read bytesize value from config object and write to combobox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_RS232KEY_BYTESIZE)))
        SetComboBox ( IDC_COMBO_BYTESIZE, iterKeys->second);

      /* Read parity value from config object and write to combobox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_RS232KEY_PARITY)))
        SetComboBox ( IDC_COMBO_PARITY, iterKeys->second);

      /* Read stopbits value from config object and write to combobox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_RS232KEY_STOPBITS)))
        SetComboBox ( IDC_COMBO_STOPBITS, iterKeys->second);

      /* Read interface exclude value from config object and write to checkbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_EXCLUDE)))
      {
        if (atoi(iterKeys->second))
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), BST_CHECKED);
         else
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_EXCLUDE), BST_UNCHECKED);
      }


      break;
      
    case eLAYER:
      
      /* Read connector enabled value from config object and write to checkbox */
      if (cmKeyMap.end() != (iterKeys = cmKeyMap.find(NXCON_COMMON_ENABLED)))
      {
        if (atoi(iterKeys->second))
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE), BST_CHECKED);
         else
          Button_SetCheck(GetDlgItem(m_hDialogWnd, IDC_CHECK_ENABLE), BST_UNCHECKED);
      }
      break;
    default:
      return;
  }
}

/////////////////////////////////////////////////////////////////////////////
/// Set ComboBox selection
/// \param iItemId ID of ComboBox item
/// \param szValue List value of combobox which should be selected
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::SetComboBox (int iItemId, const char* szValue)
{
  HWND hItemDlg = GetDlgItem( m_hDialogWnd, iItemId);
  long lData;
  for (int iItem=0; CB_ERR != (lData = (long)ComboBox_GetItemData(hItemDlg, iItem)); ++iItem)
  {
    if (lData == atoi(szValue))
    {
      ComboBox_SetCurSel(hItemDlg, iItem);
      break;
    }
  }
}

///////////////////////////////////////////////////////////////////////////
/// Read interface configuration from dialog controls and store values 
/// in the layer and interface configuration Map
///////////////////////////////////////////////////////////////////////////
CString CRS232ConfigDlg::GetSelInterfaceName( void)
{
  char szName[MAX_PATH] = "";
  ComboBox_GetText(GetDlgItem(m_hDialogWnd, IDC_COMBO_INTF), szName, sizeof(szName)/sizeof(*szName)); 
  return CString(szName);
}

/////////////////////////////////////////////////////////////////////////////
/// Write interface configuration from the interface map to the dialog controls
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::ShowSelInterfaceConfig( void)
{
  CONFIGKEY_MAP cmKeyMap;  

  if ( NXCON_NO_ERROR == m_cRS232Config.LoadInterfaceConfig( GetSelInterfaceName(), cmKeyMap))
  {
    WriteToDialogItems(eINTERFACE, cmKeyMap);
  }

  /* Setup item states */
  SetInterfaceDlgItemState();
}

/////////////////////////////////////////////////////////////////////////////
/// Show Configuration of selected interface in the dialog
/////////////////////////////////////////////////////////////////////////////
void CRS232ConfigDlg::ShowLayerConfig( void)
{
  CONFIGKEY_MAP cmKeyMap;
  if (NXCON_NO_ERROR == m_cRS232Config.LoadLayerConfig( cmKeyMap))
    WriteToDialogItems(eLAYER, cmKeyMap);

  SetLayerDlgItemState();
}

/////////////////////////////////////////////////////////////////////////////
/// Function to handle dialog actions
///   \param hDlg    Window handle of the dialog
///   \param message Window messages
///   \param wParam  Parameters associated with window message
///   \param lParam  Parameters associated with window message
///   \return TRUE when processed
/////////////////////////////////////////////////////////////////////////////
INT_PTR CALLBACK CRS232ConfigDlg::DialogProc (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  UNREFERENCED_PARAMETER(lParam);

  switch (message)
  {
    case WM_INITDIALOG:
      return TRUE;

    case WM_NOTIFY:
      switch (LOWORD(wParam))
      {
        case IDC_SPIN_KEEPALIVE:
        case IDC_SPIN_SEND:
        case IDC_SPIN_RESET:
          if (UDN_DELTAPOS == ((NMHDR*)lParam)->code)
          { 
            CONFIGKEY_MAP cmKeyMap;
            CString csName = GetSelInterfaceName();
            ReadFromDialogItems( eINTERFACE, cmKeyMap);
            SetInterfaceDlgItemState();
            m_cRS232Config.Lock();
            if (NXCON_NO_ERROR != m_cRS232Config.StoreInterfaceConfig(csName, cmKeyMap))
              Edit_Undo(GetDlgItem(m_hDialogWnd, LOWORD(wParam)));
            m_cRS232Config.Unlock();
          }
          return TRUE;
      }
      break;

    case WM_COMMAND:
      switch (LOWORD(wParam))
      {
        case IDC_COMBO_INTF:
          /* Changed interface in list */
          if (CBN_SELCHANGE == (HIWORD(wParam))) 
          { 
            m_cRS232Config.Lock();
            ShowSelInterfaceConfig();
            m_cRS232Config.Unlock();
          }
          return TRUE;
        
        case IDC_CHECK_ENABLE:
          if (BN_CLICKED == (HIWORD(wParam))) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            ReadFromDialogItems( eLAYER, cmKeyMap);
            SetLayerDlgItemState();
            m_cRS232Config.Lock();
            m_cRS232Config.StoreLayerConfig(cmKeyMap);
            m_cRS232Config.Unlock();
          }
          return TRUE;

        case IDC_CHECK_EXCLUDE:
          if (BN_CLICKED == (HIWORD(wParam))) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            ReadFromDialogItems( eINTERFACE, cmKeyMap);
            SetInterfaceDlgItemState();
            CString csName = GetSelInterfaceName();
            m_cRS232Config.Lock();
            m_cRS232Config.StoreInterfaceConfig(csName, cmKeyMap);
            m_cRS232Config.Unlock();
          }
          return TRUE;

        case IDC_COMBO_BAUDRATE:
        case IDC_COMBO_BYTESIZE:
        case IDC_COMBO_STOPBITS:
        case IDC_COMBO_PARITY:
          /* Selection changed -> store in config object */
          if (CBN_SELCHANGE == (HIWORD(wParam))) 
          { 
            CONFIGKEY_MAP cmKeyMap;
            ReadFromDialogItems( eINTERFACE, cmKeyMap);
            SetInterfaceDlgItemState();
            CString csName = GetSelInterfaceName();
            m_cRS232Config.Lock();            
            m_cRS232Config.StoreInterfaceConfig(csName, cmKeyMap);
            m_cRS232Config.Unlock();
          }
          return TRUE;

        case IDC_EDIT_KEEPALIVE:
        case IDC_EDIT_SEND:
        case IDC_EDIT_RESET:
          if (EN_KILLFOCUS == HIWORD(wParam))
          { 
            CONFIGKEY_MAP cmKeyMap;
            CString csName = GetSelInterfaceName();
            ReadFromDialogItems( eINTERFACE, cmKeyMap);
            SetInterfaceDlgItemState();
            m_cRS232Config.Lock();
            if (NXCON_NO_ERROR != m_cRS232Config.StoreInterfaceConfig(csName, cmKeyMap))
              Edit_Undo(GetDlgItem(m_hDialogWnd, LOWORD(wParam)));
            m_cRS232Config.Unlock();
          }
          return TRUE;
      }

    case WM_HSCROLL:
        return 0;

    case WM_CLOSE:
        EndDialog(hDlg, TRUE);
        return TRUE;
  }
  
  //return ::DefWindowProc( hDlg, message, wParam, lParam);
  return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
/// Wrapper function to handle dialog actions
/// \param hDlg    Window handle of the dialog
/// \param message Window messages
/// \param wParam  Parameters associated with window message
/// \param lParam  Parameters associated with window message
/////////////////////////////////////////////////////////////////////////////
INT_PTR CALLBACK CRS232ConfigDlg::DialogProcStatic(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
  return(s_cInstance->DialogProc(hDlg, message, wParam, lParam));
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
